//===============================================================================
// Microsoft patterns & practices Enterprise Library
// Configuration Application Block
//===============================================================================
// Copyright  Microsoft Corporation.  All rights reserved.
// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY
// OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT
// LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
// FITNESS FOR A PARTICULAR PURPOSE.
//===============================================================================

using System;
using System.Reflection;

using Microsoft.Practices.EnterpriseLibrary.Configuration.Factory;
namespace Microsoft.Practices.EnterpriseLibrary.Configuration.Injection
{
	/// <summary>
	/// Represents the base class from which all implementations of collection property parameter attributes must derive.
	/// </summary>
    public class CollectionConverterAttribute : CollectionConverterBaseAttribute
	{	
		private readonly Type collectionType;

        public CollectionConverterAttribute(Type collectionType)
            : this(collectionType, null)
        {
        }

		/// <summary>
		/// Initialize a new instance of the <see cref="CollectionPropertyParameterAttribute"/> class with the order, the parameter type, the property name and the type of collection to create.
		/// </summary>
		/// <param name="collectionType">The type of collection to create.</param>
        public CollectionConverterAttribute(Type collectionType, Type elementFactoryType)
            : base(elementFactoryType)
		{			
			this.collectionType = collectionType;
		}

		/// <summary>
		/// Gets the type of collection to create.
		/// </summary>
		/// <value>
		/// The type of collection to create.
		/// </value>
		public Type CollectionType
		{
			get { return collectionType; }
		}

		/// <summary>
        /// Gets the <see cref="CollectionFactory"/> to use to transform the property value to the value for the parameter.
		/// </summary>
        /// <returns>The <see cref="CollectionFactory"/> to use to transform the property value to the value for the parameter.</returns>
        protected internal override IObjectFactory<TObjectToCreate, TObjectConfiguration> GetFactory<TObjectToCreate, TObjectConfiguration>(TObjectConfiguration objectConfiguration)
		{
            Type elementType = typeof(TObjectToCreate).GetGenericArguments()[0];
            Type elementConfigurationType = typeof(TObjectConfiguration).GetGenericArguments()[0];

            Type collectionFactory = typeof(CollectionFactory<,,,>);
            collectionFactory = collectionFactory.MakeGenericType(typeof(TObjectToCreate), elementType, elementConfigurationType, typeof(TObjectConfiguration));

            object elementFactory = GetElementFactoryGeneric(elementType, elementConfigurationType);

            return (IObjectFactory<TObjectToCreate, TObjectConfiguration>)Activator.CreateInstance(collectionFactory, collectionType, elementFactory);
		}
	}
}
